﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions; 
using System.Linq.Dynamic;
using System.Data.Linq;
using System.Text;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager.Utilities; 

namespace VeteransAffairs.Registries.BusinessManager
{
    public enum EfrWorkCaseType
    {
        None = 0,
        BiologicalMonitoring = 3,
        FragmentAnalysis = 4
    }

    public enum LetterType
    {
        None = 0,
        PatientBioMonitoring = 11,
        ProviderBioMonitoring = 12,
        PatientFragment = 13,
        ProviderFragment = 14
    }


    public class LabResultsIntrprtnManager : BaseBO
    {
        public LabResultsIntrprtnManager() 
        {
            _defaultSortField = "WKF_CASE_ID desc";
        }

        private void SetLoadwithForLabResults(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.WKF_CASE);
            //lo.LoadWith<WKF_CASE>(e => e.PATIENT); 
            lo.LoadWith<WKF_CASE>(e => e.REFERRAL); //To filter the results for EFR
            lo.LoadWith<REFERRAL>(e => e.STD_INSTITUTION);
            lo.LoadWith<REFERRAL>(e => e.PATIENT);
            lo.LoadWith<WKF_CASE>(e => e.STD_WKFCASETYPE);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYTYPE);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYST);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private IQueryable<WKF_CASE_ACTIVITY> SelectByStatusLinqFilter(string value, WorkflowCaseActivityStatus status)
        {
            var results = (from ints in _db.WKF_CASE_ACTIVITies.Where(act => act.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.Interpretation)
                               && act.ARCHIVED_FLAG == false
                               && act.STD_WKFACTIVITYSTS_ID == Convert.ToInt16(status)
                               && act.WKF_CASE.WKF_CASE_ACTIVITies.Any(a => a.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult)
                               && a.STD_WKFACTIVITYSTS_ID == Convert.ToInt16(WorkflowCaseActivityStatus.Accepted)))
                                    select ints);  
                               
            string fieldSearch = "LAST_NAME";

            var param = Expression.Parameter(typeof(WKF_CASE_ACTIVITY), "PATIENT");

            // e => e.WKF_CASE.PATIENT.LAST_NAME.Contains()
            var name = Expression.PropertyOrField(Expression.PropertyOrField(Expression.PropertyOrField(param, "WKF_CASE"), "PATIENT"), fieldSearch);
            var search = Expression.Constant(value, typeof(string));

            var body = Expression.Call(name, "Contains", null, search);

            Expression<Func<WKF_CASE_ACTIVITY, bool>> lambda;
            if (String.IsNullOrEmpty(value))
            {
                lambda = x => true;
            }
            else
            {
                lambda = Expression.Lambda<Func<WKF_CASE_ACTIVITY, bool>>(body, param);
            }

            return results.Where(lambda); 

        }

        /// <summary>
        /// Delete an interpretation letter by work flow case activity ID
        /// </summary>
        /// <param name="workflowActivityId"></param>
        /// <returns></returns>
        public int DeleteInterpretationLetter(int workflowActivityId)
        {
            int returnStatus = 0;
            if (workflowActivityId > 0)
            {
                using (_db = GetDataContext())
                {
                    INTERPRETATION_LETTER letter = (from e in _db.INTERPRETATION_LETTERs
                                                    where e.WKF_CASE_ACTIVITY_ID == workflowActivityId
                                                    select e).FirstOrDefault();
                    _db.INTERPRETATION_LETTERs.DeleteOnSubmit(letter);

                    try
                    {
                        _db.SubmitChanges(ConflictMode.ContinueOnConflict);
                        returnStatus = 0;

                    }
                    catch (ChangeConflictException)
                    {
                        _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                        returnStatus = 0;
                    }
                    catch
                    {
                        returnStatus = -1;
                    }
                }
            }
            return returnStatus;
        }
        /// <summary>
        /// Update letter batch printing status
        /// </summary>
        /// <param name="InterpretationLetterId"></param>
        public int UpdateLetterBatchPrintingStatus(LETTER_BATCH_PRINTING_STATUS batchPrintStatus)
        {
            int returnStatus = 0;

            if (batchPrintStatus != null)
            {
                using (_db = GetDataContext())
                {
                    _db.DeferredLoadingEnabled = false;
                    batchPrintStatus.SynchroniseWithDataContext(_db);
                    ChangeSet changeSet = _db.GetChangeSet();
                    if (changeSet.Deletes.Count > 0 || changeSet.Inserts.Count > 0
                          || changeSet.Updates.Count > 0)
                    {
                        //if changes present then submit changes
                        try
                        {
                            _db.SubmitChanges(ConflictMode.ContinueOnConflict);
                        }
                        catch (ChangeConflictException)
                        {
                            _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                        }
                        catch
                        {
                            returnStatus = -1;
                        }
                    }
                }
            }

            return returnStatus;
        }

        /// <summary>
        /// Add an interpretation letter to print queue
        /// </summary>
        /// <param name="InterpretationLetterId"></param>
        public int UpdateInterpretationLetterPrintQueue(INTERPRETATION_LETTER letter, 
            bool printQueue)
        {
            int returnStatus = 0;

            if (letter != null)
            {
                if (letter.PRINT_QUEUE != printQueue)
                {
                    letter.SetAsChangeTrackingRoot();
                    letter.PRINT_QUEUE = printQueue;
                    letter.SetAsUpdateOnSubmit();

                    using (_db = GetDataContext())
                    {
                        _db.DeferredLoadingEnabled = false;
                        letter.SynchroniseWithDataContext(_db);
                        ChangeSet changeSet = _db.GetChangeSet();
                        if (changeSet.Deletes.Count > 0 || changeSet.Inserts.Count > 0
                              || changeSet.Updates.Count > 0)
                        {
                            //if changes present then submit changes
                            try
                            {
                                _db.SubmitChanges(ConflictMode.ContinueOnConflict);
                            }
                            catch (ChangeConflictException)
                            {
                                _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                            }
                            catch
                            {
                                returnStatus = -1;
                            }
                        }
                    }
                }
            }

            return returnStatus;
        }

        /// <summary>
        /// Update an interpretation letter - insert, update or delete
        /// </summary>
        /// <param name="InterpretationLetterId"></param>
        public int UpdateInterpretationLetter(INTERPRETATION_LETTER letter)
        {
            int returnStatus = 0;

            if (letter != null)
            {
                using (_db = GetDataContext())
                {
                    _db.DeferredLoadingEnabled = false;
                    letter.SynchroniseWithDataContext(_db);
                    ChangeSet changeSet = _db.GetChangeSet();
                    if (changeSet.Deletes.Count > 0 || changeSet.Inserts.Count > 0
                          || changeSet.Updates.Count > 0)
                    {
                        //if changes present then submit changes
                        try
                        {
                            _db.SubmitChanges(ConflictMode.ContinueOnConflict);
                        }
                        catch (ChangeConflictException)
                        {
                            _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                        }
                        catch
                        {
                            returnStatus = -1;
                        }
                    }
                }
            }

            return returnStatus;
        }

        public LETTER_BATCH_PRINTING_STATUS GetBatchPrintingStatus()
        {
            LETTER_BATCH_PRINTING_STATUS status = null;

            using (_db = GetDataContext())
            {
                status = (from e in _db.LETTER_BATCH_PRINTING_STATUS
                        select e).FirstOrDefault();
            }

            return status;
        }
        
        public List<INTERPRETATION_LETTER> GetAllInterpretationLettersInPrintQueue(int type)
        {
            List<INTERPRETATION_LETTER> lettersInQueue = null;

            using (_db = GetDataContext())
            {
                SetLoadWithForInterpretationLetters(_db);
                lettersInQueue = (from e in _db.INTERPRETATION_LETTERs
                                       where e.PRINT_QUEUE == true
                                            && e.WKF_CASE_ACTIVITY.WKF_CASE.STD_WKFCASETYPE_ID == type
                                       select e).ToList();
            }

            return lettersInQueue;
        }

        
        public IEnumerable<LabResultInterpetation> GetInterpretationActivities(string searchValue, WorkflowCaseActivityStatus status, string sort, int maxRows, int startRow)
        {
            if (string.IsNullOrEmpty(sort))
                sort = _defaultSortField;
            using (_db = GetDataContext())
            {
                SetLoadwithForLabResults(_db);

                var entities = SelectByStatusLinqFilter(searchValue, status).OrderBy(sort).Skip(startRow).Take(maxRows).Distinct().ToList(); 
                
                return (entities.Select(t => new LabResultInterpetation
                             {
                                 CaseId = t.WKF_CASE_ID,
                                 ActivityId = t.WKF_CASE_ACTIVITY_ID,
                                 AcceptedDate =  t.UPDATED, //last update to lab results(UPDATED) is same as interpretation New (CREATED)
                                 PatientName = t.WKF_CASE.REFERRAL.PATIENT == null ? String.Empty : t.WKF_CASE.REFERRAL.PATIENT.FullName,
                                 CaseTypeId = t.WKF_CASE.STD_WKFCASETYPE.ID,
                                 CaseType = t.WKF_CASE.STD_WKFCASETYPE.Name,
                                 StationNumber = t.WKF_CASE.REFERRAL.STD_INSTITUTION.STATIONNUMBER,
                                 Institution = t.WKF_CASE.REFERRAL.STD_INSTITUTION.NAME
                                 //IntActivityId = t.WKF_CASE_ACTIVITY_ID //Commented to avoid duplicates
                             }).Distinct());
            }
        }

        public int GetInterpretationActivitiesCount(string searchValue, WorkflowCaseActivityStatus status)
        {
            using (_db = GetDataContext())
            {
                return SelectByStatusLinqFilter(searchValue, status).Count();
            }
        }

        private void SetLoadwithForLabResultDetails(RegistriesDataAccess db, int caseID)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<WKF_CASE>(e => e.REFERRAL); //To filter the results for EFR
            lo.LoadWith<REFERRAL>(e => e.PATIENT);
            lo.LoadWith<REFERRAL>(e => e.STD_INSTITUTION);
            lo.LoadWith<STD_INSTITUTION>(e => e.STD_COUNTRY);
            lo.LoadWith<STD_INSTITUTION>(e => e.STD_STATE);
            lo.LoadWith<PATIENT>(e => e.REGISTRY_DEMOGRAPHICs);
            lo.AssociateWith<PATIENT>(e => e.REGISTRY_DEMOGRAPHICs.Where(d => d.STD_REGISTRY_ID == UserMgr.RegistryId));
            lo.LoadWith<PATIENT>(e => e.STD_GENDER);
            lo.LoadWith<WKF_CASE>(e => e.WKF_CASE_ACTIVITies);
            lo.AssociateWith<WKF_CASE>(e => e.WKF_CASE_ACTIVITies.Where(
                a => a.WKF_CASE_ID == caseID && a.ARCHIVED_FLAG == false 
               && ((a.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult) 
                   && a.STD_WKFACTIVITYSTS_ID == Convert.ToInt16(WorkflowCaseActivityStatus.Accepted)) 
                || a.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.Interpretation))));
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.SPECIMEN_ANALYSIs);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYST);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.INTERPRETATION_LETTERs);
            lo.LoadWith<SPECIMEN_ANALYSI>(e => e.ANALYSIS_DETAILs);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_ANALYSISMETHOD);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_ANALYTETYPE);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_TEST_NAME);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_UNIT);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_UNIT1);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_UNIT2);
            
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }
        
        /// <summary>
        /// Gets lab result details for a selected activity Id
        /// </summary>
        /// <param name="activityId"></param>
        /// <returns></returns>
        public WKF_CASE GetLabInterpretationDetails(int caseId)
        {
            WorkflowManager Wfm = new WorkflowManager();
            WorkFlowTypes WfType = Wfm.GetWorkflowCaseType(caseId);

            using (_db = GetDataContext())
            {
                if (WfType == WorkFlowTypes.Biomonitoring)
                    SetLoadwithForLabResultDetails(_db, caseId);
                if (WfType == WorkFlowTypes.FragmentAnalysis)
                    SetLoadwithForFragLabResultDetails(_db, caseId);

                return (from t in _db.WKF_CASEs
                        where t.WKF_CASE_ID == caseId
                        select t).FirstOrDefault();
            }
        }

        private void SetLoadWithForInterpretationLetters(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<INTERPRETATION_LETTER>(e => e.WKF_CASE_ACTIVITY);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.WKF_CASE);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadwithForFragLabResultDetails(RegistriesDataAccess db, int caseId)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<WKF_CASE>(e => e.REFERRAL); //To filter the results for EFR
            lo.LoadWith<REFERRAL>(e => e.PATIENT);
            lo.LoadWith<REFERRAL>(e => e.STD_INSTITUTION); 
            lo.LoadWith<PATIENT>(e => e.REGISTRY_DEMOGRAPHICs);
            lo.AssociateWith<PATIENT>(e => e.REGISTRY_DEMOGRAPHICs.Where(d => d.STD_REGISTRY_ID == UserMgr.RegistryId));
            lo.LoadWith<PATIENT>(e => e.STD_GENDER);
            lo.LoadWith<WKF_CASE>(e => e.WKF_CASE_ACTIVITies);
            lo.AssociateWith<WKF_CASE>(e => e.WKF_CASE_ACTIVITies.Where(a => a.WKF_CASE_ID == caseId && (a.ARCHIVED_FLAG == false) && (a.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult) && a.STD_WKFACTIVITYSTS_ID == Convert.ToInt16(WorkflowCaseActivityStatus.Accepted)) || a.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.Interpretation) || a.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabOrder)));
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.FRAGMENT_ANALYSIs);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.LAB_ORDERs); 
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.STD_WKFACTIVITYST);
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.INTERPRETATION_LETTERs);
            lo.LoadWith<FRAGMENT_ANALYSI>(e => e.ANALYSIS_DETAILs);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_ANALYSISMETHOD);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_FRAGMENT_ANALYSIS_TYPE);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_ANALYTETYPE); 
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        /// <summary>
        /// Saves the interpretation details
        /// </summary>
        /// <param name="patient"></param>
        public void UpdateInterpretationActivity(PATIENT patient)
        {
            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;
                patient.SynchroniseWithDataContext(_db, true);
                try
                {
                    _db.SubmitChanges();

                    //create an instance of the custom eventArgs in order to populate the id selected
                    BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
                    eventArgs.SavedItemId = patient.PATIENT_ID;

                    RaiseSaveEvent(this, eventArgs);
                }
                catch (ChangeConflictException)
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                }
            }
        }
        /// <summary>
        /// Updates the work flow activity status 
        /// </summary>
        /// <param name="Activity"></param>
        public void UpdateInterpretationActivity(WKF_CASE_ACTIVITY activity)
        {
            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;
                activity.SynchroniseWithDataContext(_db, true);
                try
                {
                    _db.SubmitChanges();
                }
                catch (ChangeConflictException)
                {
                    _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                }
            }
                
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="activityId - of lab result"></param>
        /// <returns></returns>
        public IQueryable<WKF_CASE_ACTIVITY> LinqGetIntrprtnActivity(int activityId)
        {
            using (_db = GetDataContext())
            {
                return (from t in _db.WKF_CASE_ACTIVITies.Where(act => act.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.LabResult)
                        && act.ARCHIVED_FLAG == false
                        && act.WKF_CASE.REFERRAL.STD_REGISTRY_ID == UserMgr.RegistryId
                        && act.STD_WKFACTIVITYSTS_ID == Convert.ToInt16(WorkflowCaseActivityStatus.Accepted))
                        join t1 in _db.WKF_CASE_ACTIVITies on t.WKF_CASE_ID equals t1.WKF_CASE_ID
                        where t1.STD_WKFACTIVITYTYPE_ID == Convert.ToInt16(WorkFlowActivityTypes.Interpretation)
                        && t1.ARCHIVED_FLAG == false
                        && t1.WKF_CASE.REFERRAL.STD_REGISTRY_ID == UserMgr.RegistryId
                        && t1.STD_WKFACTIVITYSTS_ID == Convert.ToInt16(WorkflowCaseActivityStatus.New) select t1);
            }

        }
        #region Letters

        #region Everything except analytes
        public IEnumerable<Letter> GetLetterContent(int intActivityId, string rptType)
        {
            if(rptType == "0") //Patient
                return GetPatientLetterContent(intActivityId);
            else if (rptType == "1")
               return GetProviderLetterContent(intActivityId);
            return null;
        }

        private IEnumerable<Letter> GetPatientLetterContent(int intActivityId)
        {
            using (_db = GetDataContext())
            {
                return (from t in _db.INTERPRETATION_LETTERs
                        where t.WKF_CASE_ACTIVITY_ID == intActivityId
                        select new Letter
                             {
                                 Name = t.PATIENT_NAME,
                                 AddressPart1 = t.PATIENT_ADDR1,
                                 AddressPart2 = t.PATIENT_ADDR2,
                                 AddressPart3 = t.PATIENT_ADDR3,
                                 City = t.PATIENT_CITY,
                                 Country = t.PATIENT_COUNTRY,
                                 State = t.PATIENT_STATE,
                                 Zip = t.PATIENT_POSTAL,
                                 InterpretationText = t.PATIENT_INTERPRETATION_TEXT, 
                                 Gender = GetPatientGender(intActivityId),
                                 Last4SSN = GetLast4DigitsOfSSN(intActivityId),
                                 PatientLastName = GetPatientLastName(intActivityId),
                                 WorkflowType = GetWorkflowType(intActivityId)
                             }).ToList();
            }
        }

        private IEnumerable<Letter> GetProviderLetterContent(int intActivityId)
        {
            using (_db = GetDataContext())
            {
                IEnumerable<Letter> letters = (from t in _db.INTERPRETATION_LETTERs
                        where t.WKF_CASE_ACTIVITY_ID == intActivityId
                        select new Letter
                        {
                            Name = t.PROVIDER_NAME,
                            AddressPart1 = t.PROVIDER_ADDR1,
                            AddressPart2 = t.PROVIDER_ADDR2,
                            AddressPart3 = t.PROVIDER_ADDR3,
                            City = t.PROVIDER_CITY,
                            Country = t.PROVIDER_COUNTRY,
                            State = t.PROVIDER_STATE,
                            Zip = t.PROVIDER_POSTAL,
                            InterpretationText = t.PATIENT_INTERPRETATION_TEXT,
                            Gender = t.PATIENT_NAME + "-" +  GetLast4DigitsOfSSN(intActivityId),
                            //Last4SSN = GetLast4DigitsOfSSN(intActivityId),
                            PatientLastName = GetPatientLastName(intActivityId),
                            WorkflowType = GetWorkflowType(intActivityId)
                        }).ToList();
                return letters;
            }
        }

        private string GetLast4DigitsOfSSN(int intActivityId)
        {
            using (_db = GetDataContext())
            {
                DataLoadOptions lo = new DataLoadOptions();
                lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.WKF_CASE);
                lo.LoadWith<WKF_CASE>(e => e.REFERRAL);
                lo.LoadWith<REFERRAL>(e => e.PATIENT);
                return (_db.WKF_CASE_ACTIVITies.Where(e => e.WKF_CASE_ACTIVITY_ID == intActivityId).Select(t => t.WKF_CASE.REFERRAL.PATIENT.Snum)).FirstOrDefault().LastChars(4);
            }
        }

        private string GetPatientGender(int activityId)
        {
            using (_db = GetDataContext())
            {
                DataLoadOptions lo = new DataLoadOptions();
                lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.WKF_CASE);
                lo.LoadWith<WKF_CASE>(e => e.REFERRAL);
                lo.LoadWith<REFERRAL>(e => e.PATIENT);

                return  (_db.WKF_CASE_ACTIVITies.Where(e => e.WKF_CASE_ACTIVITY_ID == activityId).Select(t => t.WKF_CASE.REFERRAL.PATIENT.FIRST_NAME + " " + t.WKF_CASE.REFERRAL.PATIENT.LAST_NAME)).FirstOrDefault();
            }
        }

        private string GetPatientLastName(int activityId)
        {
            using (_db = GetDataContext())
            {
                DataLoadOptions lo = new DataLoadOptions();
                lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.WKF_CASE);
                lo.LoadWith<WKF_CASE>(e => e.REFERRAL);
                lo.LoadWith<REFERRAL>(e => e.PATIENT);

                return (_db.WKF_CASE_ACTIVITies.Where(e => e.WKF_CASE_ACTIVITY_ID == activityId).Select(t => t.WKF_CASE.REFERRAL.PATIENT.LAST_NAME)).FirstOrDefault();
            }
        }

        private string GetWorkflowType(int activityId)
        {
            using (_db = GetDataContext())
            {
                DataLoadOptions lo = new DataLoadOptions();
                lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.WKF_CASE);
                lo.LoadWith<WKF_CASE>(e => e.STD_WKFCASETYPE);

                return (_db.WKF_CASE_ACTIVITies.Where(e => e.WKF_CASE_ACTIVITY_ID == activityId).Select(t => t.WKF_CASE.STD_WKFCASETYPE.CODE)).FirstOrDefault();
            }
        }

        #endregion

        public IEnumerable<ANALYTE_DETAILS> GetLetterAnalytes(int activityId)
        {
            WorkflowManager Wfm = new WorkflowManager();
            int AnalyteType = Wfm.GetWorkflowType(activityId);

            if (AnalyteType == Convert.ToInt16(WorkFlowTypes.Biomonitoring)) //Bio
                return GetBioAnalytes(activityId);
            else if (AnalyteType == Convert.ToInt16(WorkFlowTypes.FragmentAnalysis)) //Fragment
                return GetFragmentAnalytes(activityId);
            return null;
        }

        private IEnumerable<ANALYTE_DETAILS> GetBioAnalytes(int activityId)
        {
            using (_db = GetDataContext())
            {
                SetLoadwithForBioAnalytesForLetters(_db, activityId);
                IEnumerable<ANALYTE_DETAILS> obj = (from t in _db.ANALYSIS_DETAILs
                        where (t.SPECIMEN_ANALYSI.WKF_CASE_ACTIVITY_ID == activityId || t.FRAGMENT_ANALYSI.WKF_CASE_ACTIVITY_ID == activityId)
                        select new ANALYTE_DETAILS
                        {
                             TestMethod=t.TEST_NAME,
                             Analyte=t.ANALYTE_NAME,
                             Result=t.RESULT_TEXT,
                             //Units=t.RESULT_UNIT_NUMBER,
                             Range=t.LAB_REFERENCE_RANGE
                        }).ToList();
                return obj;
            }
        }

        private IEnumerable<ANALYTE_DETAILS> GetFragmentAnalytes(int activityId)
        {
            using (_db = GetDataContext())
            {
                SetLoadwithForFragmentAnalytesForLetters(_db, activityId);
                IEnumerable<ANALYTE_DETAILS> obj = (from t in _db.ANALYSIS_DETAILs
                        where (t.SPECIMEN_ANALYSI.WKF_CASE_ACTIVITY_ID == activityId || t.FRAGMENT_ANALYSI.WKF_CASE_ACTIVITY_ID == activityId)
                        select new ANALYTE_DETAILS
                        {
                             TestMethod = t.STD_ANALYSISMETHOD.CODE,
                             Analyte = t.ANALYTE_NAME
                             //, Result = t.RESULT_UNIT_NUMBER
                             , Range= t.LAB_REFERENCE_RANGE}
                        ).ToList();
                return obj;
            }
        }
        private void SetLoadwithForBioAnalytesForLetters(RegistriesDataAccess db, int activityId)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.SPECIMEN_ANALYSIs);
            lo.AssociateWith<WKF_CASE_ACTIVITY>(e => e.SPECIMEN_ANALYSIs.Where(s => s.WKF_CASE_ACTIVITY_ID == activityId));
            lo.LoadWith<SPECIMEN_ANALYSI>(e => e.ANALYSIS_DETAILs);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadwithForFragmentAnalytesForLetters(RegistriesDataAccess db, int activityId)
        {

            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<WKF_CASE_ACTIVITY>(e => e.FRAGMENT_ANALYSIs);
            lo.AssociateWith<WKF_CASE_ACTIVITY>(e => e.FRAGMENT_ANALYSIs.Where(s => s.WKF_CASE_ACTIVITY_ID == activityId));
            lo.LoadWith<FRAGMENT_ANALYSI>(e => e.ANALYSIS_DETAILs);
            lo.LoadWith<ANALYSIS_DETAIL>(e => e.STD_ANALYSISMETHOD);
            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

    }
        #endregion

    public class ANALYTE_DETAILS
    {
        public string TestMethod { get; set; }
        public string Analyte { get; set; }
        public string Result { get; set; }
        public string Units { get; set; }
        public string Range { get; set; }
    }

    public class LabResultInterpetation : IEquatable<LabResultInterpetation>
    {
        public DateTime AcceptedDate { get; set; }
        public int CaseId { get; set; }
        public int ActivityId { get; set; }
        public string PatientName { get; set; }
        public string StationNumber { get; set; }
        public string Institution { get; set; }
        public int CaseTypeId { get; set; }
        public string CaseType { get; set; }
        public int IntActivityId { get; set; }

        public bool Equals(LabResultInterpetation other)
        {
            return ActivityId == other.ActivityId;
        }

        public override bool Equals(object other)
        {
            LabResultInterpetation otherSpecific = other as LabResultInterpetation;

            return otherSpecific != null && ActivityId == otherSpecific.ActivityId;
        }

        public override int GetHashCode()
        {
            return ActivityId.GetHashCode();  
        }
    }

    public class Address
    {
        public string Name { get; set; }
        public string AddressPart1 { get; set; }
        public string AddressPart2 { get; set; }
        public string AddressPart3 { get; set; }
        public string City { get; set; }
        public string State { get; set; }
        public string Zip { get; set; }
        public string Country { get; set; }
        public string County { get; set; }
        public string Phone { get; set; }
        public string Facility { get; set; }
    }

    public class Letter : Address
    {
        public string InterpretationText { get; set; }
        public string LastName
        {
            get
            {
                return GetLastName(Name);
            }

        }
        public string Gender{get; set;}
        public string GetLastName(string fullName)
        {
            string[] Names = fullName.Split(new Char[] { ',' });
            return Names.Where(e => (e.Trim() != "" && e != null)).FirstOrDefault();
        }
        public string PatientLastName { get; set; }
        public string Last4SSN { get; set; }
        public string WorkflowType { get; set; }
        
    }

    public class PatientPCPInfo
    {
        public string Name { get; set; }
        public string Address1 { get; set; }
        public string Address2 { get; set; }
        public string Address3 { get; set; }
        public string City { get; set; }
        public string State { get; set; }
        public string Zip { get; set; }
        public string Country { get; set; }
        public string Phone { get; set; }
    
    }

    public class PatientVAMCInfo : PatientPCPInfo 
    {
        //public string Name { get; set; }
        public string Facility { get; set; }
        //public string Address1 { get; set; }
        //public string Address2 { get; set; }
        //public string Address3 { get; set; }
        //public string City { get; set; }
        //public string State { get; set; }
        //public string Zip { get; set; }
        //public string Country { get; set; }
        //public string Phone { get; set; }
    }

}
